"""
Strategies from Axelrod's second tournament. All strategies in this module are
prefixed by `SecondBy` to indicate that they were submitted in Axelrod's Second
tournament by the given author.
"""

from typing import List

import numpy as np

from axelrod.action import Action

from axelrod.interaction_utils import compute_final_score

from axelrod.player import Player

from axelrod.strategies.finite_state_machines import FSMPlayer

C, D = Action.C, Action.D

class SecondByWhite(Player):
    """
    Strategy submitted to Axelrod's second tournament by Edward C White (K72R)
    and came in thirteenth in that tournament.

    * Cooperate in the first ten turns.
    * If the opponent Cooperated last turn then Cooperate.
    * Otherwise Defect if and only if:
        floor(log(turn)) * opponent Defections >= turn

    Names:

    - White: [Axelrod1980b]_
    """

    name = "Second by White"
    classifier = {
        "memory_depth": float("inf"),
        "stochastic": False,
        "long_run_time": False,
        "inspects_source": False,
        "manipulates_source": False,
        "manipulates_state": False,
    }

    def strategy(self, opponent: Player) -> Action:
        turn = len(self.history) + 1

        if turn <= 10 or opponent.history[-1] == C:
            return C

        if np.floor(np.log(turn)) * opponent.defections >= turn:
            return D
        return C